<?php
if (!defined('ABSPATH')) exit;

// Licensing helpers (premium features)
function wpsl_get_license_key(): string {
    return sanitize_text_field((string)get_option('wpsl_license_key', ''));
}

function wpsl_set_license_key(string $key): void {
    update_option('wpsl_license_key', sanitize_text_field($key));
}

function wpsl_licensing_api_url(): string {
    // Licensing endpoint lives at site root /api/license.php (even if WP is under a subdirectory)
    $scheme = is_ssl() ? 'https' : 'http';
    $host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : 'localhost';
    return $scheme . '://' . $host . '/api/license.php';
}

function wpsl_verify_license_cached(): array {
    $key = wpsl_get_license_key();
    if ($key === '') return ['valid' => false, 'reason' => 'no_key'];
    $cached = get_transient('wpsl_license_status');
    if (is_array($cached)) return $cached;
    $resp = wp_remote_post(wpsl_licensing_api_url(), [
        'timeout' => 10,
        'body' => [ 'action' => 'verify', 'license_key' => $key, 'domain' => wp_parse_url(home_url(), PHP_URL_HOST) ]
    ]);
    $ok = (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) === 200);
    if (!$ok) {
        $code = is_wp_error($resp) ? 'WP_Error' : (string) wp_remote_retrieve_response_code($resp);
        $emsg = is_wp_error($resp) ? $resp->get_error_message() : '';
        $result = ['valid' => false, 'reason' => 'http_error', 'http_code' => $code, 'error_message' => $emsg];
        set_transient('wpsl_license_status', $result, 30);
        return $result;
    }
    $body = (string) wp_remote_retrieve_body($resp);
    $json = json_decode($body, true);
    $result = ['valid' => false, 'reason' => 'unknown'];
    if (is_array($json)) {
        // New API format: { ok: bool, license: { valid: bool, ... }, error?: string }
        if (isset($json['license']) && is_array($json['license'])) {
            $lic = $json['license'];
            $is_valid = !empty($lic['valid']);
            if ($is_valid) {
                $result = ['valid' => true, 'license' => $lic, 'api_ok' => (bool)($json['ok'] ?? true)];
            } else {
                $reason = (string)($lic['reason'] ?? ($json['error'] ?? 'invalid'));
                $result = ['valid' => false, 'reason' => $reason, 'license' => $lic, 'api_ok' => (bool)($json['ok'] ?? false)];
            }
        } else {
            // Legacy format: { valid: bool, reason?: string }
            if (!empty($json['valid'])) {
                $result = ['valid' => true, 'license' => $json];
            } else {
                $result = ['valid' => false, 'reason' => (string)($json['reason'] ?? $json['error'] ?? 'invalid')];
            }
        }
        // Keep raw debug snippet
        $result['raw'] = is_array($json) ? $json : [];
    } else {
        $result = ['valid' => false, 'reason' => 'bad_json'];
    }
    set_transient('wpsl_license_status', $result, 300);
    return $result;
}

function wpsl_is_premium_active(): bool {
    $st = wpsl_verify_license_cached();
    return !empty($st['valid']);
}

function wpsl_render_license_page() {
    if (!current_user_can('manage_options')) return;
    $message = '';
    if (isset($_POST['wpsl_license_nonce']) && wp_verify_nonce((string)$_POST['wpsl_license_nonce'], 'wpsl_save_license')) {
        $key = isset($_POST['wpsl_license_key']) ? sanitize_text_field((string)$_POST['wpsl_license_key']) : '';
        wpsl_set_license_key($key);
        delete_transient('wpsl_license_status');
        $st = wpsl_verify_license_cached();
        if (!empty($st['valid'])) {
            // Attempt activation (bind to this domain)
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
            $admin_email = get_option('admin_email');
            $resp = wp_remote_post(wpsl_licensing_api_url(), [
                'timeout' => 10,
                'body' => [ 'action'=>'activate', 'license_key'=>$key, 'domain'=>$domain, 'email'=>$admin_email ]
            ]);
            $ok = (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) === 200);
            if ($ok) {
                $message = 'License verified and activated.';
            } else {
                $code = is_wp_error($resp) ? 'WP_Error' : (string)wp_remote_retrieve_response_code($resp);
                $err  = is_wp_error($resp) ? $resp->get_error_message() : substr((string)wp_remote_retrieve_body($resp), 0, 200);
                $message = 'License verified. Activation attempt did not complete. (code: ' . esc_html($code) . ')';
                if ($err) { $message .= ' Details: ' . esc_html($err); }
            }
        } else {
            $reason = isset($st['reason']) ? (string)$st['reason'] : 'unknown';
            if ($reason === 'http_error') {
                $code = isset($st['http_code']) ? (string)$st['http_code'] : '';
                $emsg = isset($st['error_message']) ? (string)$st['error_message'] : '';
                $message = 'License not valid: http_error';
                if ($code !== '') $message .= ' (code: ' . esc_html($code) . ')';
                if ($emsg !== '') $message .= ' Details: ' . esc_html($emsg);
            } else {
                $message = 'License not valid: ' . esc_html($reason);
            }
        }
    }
    $st = wpsl_verify_license_cached();
    $status = !empty($st['valid']) ? 'Active' : 'Inactive';
    $color = !empty($st['valid']) ? 'color:#00a32a' : 'color:#d63638';
    $lic  = isset($st['license']) && is_array($st['license']) ? $st['license'] : [];
    ?>
    <div class="wrap">
        <h1>License</h1>
        <?php if ($message): ?><div class="updated notice"><p><?php echo esc_html($message); ?></p></div><?php endif; ?>
        <form method="post">
            <?php wp_nonce_field('wpsl_save_license', 'wpsl_license_nonce'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="wpsl_license_key">License Key</label></th>
                    <td>
                        <input type="text" id="wpsl_license_key" name="wpsl_license_key" class="regular-text" value="<?php echo esc_attr(wpsl_get_license_key()); ?>" placeholder="XXXX-XXXX-XXXX" />
                        <p class="description">Enter your license key to unlock premium features.</p>
                        <p><strong>Status:</strong> <span style="<?php echo esc_attr($color); ?>"><?php echo esc_html($status); ?></span></p>
                        <?php if (!empty($lic)): ?>
                            <table class="form-table" role="presentation" style="margin-top:8px;">
                                <tr>
                                    <th scope="row">Plan</th>
                                    <td><?php echo esc_html($lic['plan'] ?? ''); ?></td>
                                </tr>
                                <tr>
                                    <th scope="row">Expires</th>
                                    <td><?php echo esc_html($lic['expires_at'] ?? ''); ?></td>
                                </tr>
                                <tr>
                                    <th scope="row">Sites</th>
                                    <td><?php echo intval($lic['sites_used'] ?? 0); ?> / <?php echo intval($lic['sites_allowed'] ?? 0); ?></td>
                                </tr>
                                <?php if (isset($lic['bound_to_domain'])): ?>
                                <tr>
                                    <th scope="row">Bound to this domain</th>
                                    <td><?php echo ($lic['bound_to_domain'] ? 'Yes' : 'No'); ?></td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        <?php else: ?>
                            <?php if (!empty($st['reason'])): ?>
                                <p class="description">Reason: <?php echo esc_html($st['reason']); ?></p>
                                <?php if (!empty($st['raw']['error'])): ?>
                                    <p class="description">API Error: <?php echo esc_html((string)$st['raw']['error']); ?></p>
                                <?php endif; ?>
                            <?php endif; ?>
                        <?php endif; ?>
                        <?php $domain = wp_parse_url(home_url(), PHP_URL_HOST); $admin_email = get_option('admin_email'); $buy_url = 'https://castio.live/purchase?domain=' . rawurlencode((string)$domain) . '&email=' . rawurlencode((string)$admin_email); ?>
                        <p><a class="button" href="<?php echo esc_url( $buy_url ); ?>" target="_blank" rel="noopener">Purchase Premium</a></p>
                    </td>
                </tr>
            </table>
            <p><button class="button button-primary" type="submit">Save &amp; Verify</button></p>
        </form>
    </div>
    <?php
}
