<?php
if (!defined('ABSPATH')) exit;

function wpsl_render_settings_page($plugin_instance)
{
	if (!current_user_can('manage_options')) return;
	if (isset($_POST['wpsl_save_settings']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$premium = wpsl_is_premium_active();
		$pk = isset($_POST['wpsl_stripe_pk']) ? sanitize_text_field(wp_unslash($_POST['wpsl_stripe_pk'])) : '';
		$sk = isset($_POST['wpsl_stripe_sk']) ? sanitize_text_field(wp_unslash($_POST['wpsl_stripe_sk'])) : '';
		$cur = isset($_POST['wpsl_default_currency']) ? sanitize_text_field(wp_unslash($_POST['wpsl_default_currency'])) : 'usd';
		$hls_latency = isset($_POST['wpsl_hls_latency']) ? intval($_POST['wpsl_hls_latency']) : 4;
		$poll_latency = isset($_POST['wpsl_default_poll']) ? (float)$_POST['wpsl_default_poll'] : 4.0;
		$max_minutes = isset($_POST['wpsl_max_minutes']) ? intval($_POST['wpsl_max_minutes']) : 120;
		$invite_enabled = isset($_POST['wpsl_invite_enabled']) ? '1' : '0';
		update_option('wpsl_stripe_pk', $pk);
		update_option('wpsl_stripe_sk', $sk);
		update_option('wpsl_default_currency', $cur);
		// Enforce defaults and premium-only editability
		$hls_latency = max(1, min(6, $hls_latency));
		$poll_latency = max(0.5, min(5.0, $poll_latency));
		if (!$premium) {
			$hls_latency = 4;
			$poll_latency = 4.0;
		}
		update_option('wpsl_hls_latency', $hls_latency);
		update_option('wpsl_default_poll', $poll_latency);
		// clamp max length 1..720 minutes (12h)
		$max_minutes = max(1, min(720, $max_minutes));
		update_option('wpsl_max_minutes', $max_minutes);
		update_option('wpsl_invite_enabled', $invite_enabled);
		if (isset($_POST['wpsl_invite_subject'])) {
			$invite_subject = sanitize_text_field(wp_unslash($_POST['wpsl_invite_subject']));
			if ($invite_subject !== '') update_option('wpsl_invite_subject', $invite_subject); else delete_option('wpsl_invite_subject');
		}
		if (isset($_POST['wpsl_invite_template'])) {
			$invite_template = wp_kses_post(wp_unslash($_POST['wpsl_invite_template']));
			if ($invite_template !== '') update_option('wpsl_invite_template', $invite_template); else delete_option('wpsl_invite_template');
		}
		echo '<div class="updated notice"><p>Settings saved.</p></div>';
		if (function_exists('flush_rewrite_rules')) {
			flush_rewrite_rules();
		}
	}

	// Create Live Channel Page on demand
	if (isset($_POST['wpsl_create_live_page']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$page_id = wpsl_ensure_live_page($plugin_instance, true);
		if ($page_id) {
			$link = get_permalink($page_id);
			echo '<div class="updated notice"><p>Live channel page ready: <a target="_blank" rel="noopener" href="' . esc_url($link) . '">' . esc_html($link) . '</a></p></div>';
		} else {
			echo '<div class="error notice"><p>Could not create Live channel page.</p></div>';
		}
	}

	// Add pages to main menu on demand
	if (isset($_POST['wpsl_add_live_to_menu']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$pid = wpsl_ensure_live_page($plugin_instance, false);
		list($ok, $msg) = wpsl_add_page_to_main_menu($pid);
		echo '<div class="' . ($ok ? 'updated' : 'error') . ' notice"><p>' . esc_html($msg) . '</p></div>';
	}
	if (isset($_POST['wpsl_add_videos_to_menu']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$pid = wpsl_ensure_streams_page($plugin_instance, false);
		list($ok, $msg) = wpsl_add_page_to_main_menu($pid);
		echo '<div class="' . ($ok ? 'updated' : 'error') . ' notice"><p>' . esc_html($msg) . '</p></div>';
	}
	if (isset($_POST['wpsl_add_purchases_to_menu']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$pid = wpsl_ensure_purchases_page($plugin_instance, false);
		list($ok, $msg) = wpsl_add_page_to_main_menu($pid);
		echo '<div class="' . ($ok ? 'updated' : 'error') . ' notice"><p>' . esc_html($msg) . '</p></div>';
	}

	// Create Videos Page on demand
	if (isset($_POST['wpsl_create_videos_page']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$page_id = wpsl_ensure_streams_page($plugin_instance, true);
		if ($page_id) {
			$link = get_permalink($page_id);
			echo '<div class="updated notice"><p>Videos page ready: <a target="_blank" rel="noopener" href="' . esc_url($link) . '">' . esc_html($link) . '</a></p></div>';
		} else {
			echo '<div class="error notice"><p>Could not create Videos page.</p></div>';
		}
	}

	// Create Purchases Page on demand
	if (isset($_POST['wpsl_create_purchases_page']) && isset($_POST['wpsl_settings_nonce']) && wp_verify_nonce((string)$_POST['wpsl_settings_nonce'], 'wpsl_save_settings')) {
		$page_id = wpsl_ensure_purchases_page($plugin_instance, true);
		if ($page_id) {
			$link = get_permalink($page_id);
			echo '<div class="updated notice"><p>Purchases page ready: <a target="_blank" rel="noopener" href="' . esc_url($link) . '">' . esc_html($link) . '</a></p></div>';
		} else {
			echo '<div class="error notice"><p>Could not create Purchases page.</p></div>';
		}
	}

	$default_tpl = '<p>Hello,</p><p>You are invited to join our live session: <strong>{{stream_title}}</strong>.</p><p>Starts at: {{start_time}}</p><p>Join here: <a href="{{viewer_url}}">{{viewer_url}}</a></p><p>See you there!</p>';
	$invite_subject_val = wpsl_get_option('wpsl_invite_subject', 'Live Stream Invitation');
	$invite_template_val = wpsl_get_option('wpsl_invite_template', $default_tpl);
	$invite_enabled_val = wpsl_get_option('wpsl_invite_enabled', '1');
	?>
	<div class="wrap">
		<h1>Castio Live Settings</h1>
		<style>
			.wpsl-help {
				display: inline-block;
				margin-left: 6px;
				color: #2271b1;
				position: relative;
				cursor: help
			}

			.wpsl-help::before {
				font-family: dashicons;
				content: "\f223";
				font-size: 16px;
				line-height: 1;
				vertical-align: middle
			}

			.wpsl-help:hover::before {
				color: #135e96
			}

			.wpsl-help:hover::after {
				content: attr(data-tip);
				position: absolute;
				left: 22px;
				top: -6px;
				background: #111;
				color: #fff;
				padding: 8px 10px;
				border-radius: 6px;
				box-shadow: 0 8px 18px rgba(0, 0, 0, .15);
				width: 300px;
				max-width: 60vw;
				font-size: 12px;
				line-height: 1.45;
				z-index: 9999
			}
		</style>
		<form method="post">
			<?php wp_nonce_field('wpsl_save_settings', 'wpsl_settings_nonce'); ?>
			<div class="nav-tab-wrapper" style="margin-top:8px;">
				<a href="#" class="nav-tab nav-tab-active" data-wpsl-tab="streaming">Streaming</a>
				<a href="#" class="nav-tab" data-wpsl-tab="payments">Payments</a>
				<a href="#" class="nav-tab" data-wpsl-tab="pages">Pages</a>
				<a href="#" class="nav-tab" data-wpsl-tab="invitation">Invitation</a>
			</div>
			<table class="form-table" role="presentation">
				<tbody data-wpsl-tab-panel="streaming">
				<tr>
					<th scope="row"><label for="wpsl-hls-latency">HLS Latency (seconds)</label> <span class="wpsl-help"
																									  data-tip="Lower latency means faster live playback but can increase CPU/network usage and stutter. 2–4s is a good balance."></span>
					</th>
					<td>
						<?php $premium = wpsl_is_premium_active();
						$hls_val = (int)get_option('wpsl_hls_latency', 4);
						if (!$premium) $hls_val = 4; ?>
						<input type="number" id="wpsl-hls-latency" name="wpsl_hls_latency" class="small-text" min="1" max="6" step="1"
							   value="<?php echo $hls_val; ?>" <?php echo $premium ? '' : 'disabled'; ?> />
						<span class="description">Lower = faster but more CPU/network and potential stutter. Recommended 2&ndash;4s.<?php if (!$premium) echo ' (Premium required to change; default 4s)'; ?></span>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wpsl-max-minutes">Max video length (minutes)</label> <span class="wpsl-help"
																										   data-tip="Upper bound for a single live or recorded video duration. Used for guidance and enforcement in future updates."></span>
					</th>
					<td>
						<?php $max_minutes = (int)get_option('wpsl_max_minutes', 120); ?>
						<input type="number" id="wpsl-max-minutes" name="wpsl_max_minutes" class="small-text" min="1" max="720" step="1" value="<?php echo esc_attr($max_minutes); ?>"/>
						<span class="description">Default: 120 minutes. Range 1&ndash;720 (12 hours).</span>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wpsl-default-poll">Default Polling Latency (seconds)</label> <span class="wpsl-help"
																												   data-tip="How often the viewer polls for new chat messages. Lower is more responsive but increases server load."></span>
					</th>
					<td>
						<?php $premium = wpsl_is_premium_active();
						$poll_val = (float)get_option('wpsl_default_poll', 4.0);
						if (!$premium) $poll_val = 4.0; ?>
						<input type="number" id="wpsl-default-poll" name="wpsl_default_poll" class="small-text" min="0.5" max="5" step="0.1"
							   value="<?php echo esc_attr($poll_val); ?>" <?php echo $premium ? '' : 'disabled'; ?> />
						<span class="description">Chat polling interval for viewers. Lower is more responsive but adds server load.<?php if (!$premium) echo ' (Premium required to change; default 4s)'; ?></span>
					</td>
				</tr>
				</tbody>
				<tbody data-wpsl-tab-panel="payments" style="display:none;">
				<tr>
					<th scope="row"><label for="wpsl-stripe-pk">Stripe Publishable Key</label> <span class="wpsl-help"
																									 data-tip="Use Stripe test keys while testing (prefix pk_test_) and switch to live keys (pk_live_) before launch. Find in Stripe Dashboard → Developers → API Keys."></span>
					</th>
					<td>
						<input type="text" id="wpsl-stripe-pk" name="wpsl_stripe_pk" class="regular-text" value="<?php echo esc_attr(wpsl_get_option('wpsl_stripe_pk')); ?>"
							   title="Find in Stripe Dashboard &rarr; Developers &rarr; API Keys &rarr; Publishable key (starts with 'pk_')"/>
						<p class="description">Stripe Dashboard &rarr; Developers &rarr; API Keys &rarr; Publishable key (starts with <code>pk_</code>).</p>
						<p class="description">Tip: Use <em>test</em> keys while testing and switch to <em>live</em> keys before launch.</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wpsl-stripe-sk">Stripe Secret Key</label> <span class="wpsl-help"
																								data-tip="Your server-side key (sk_...). No webhooks required; the plugin verifies the Checkout session on return."></span>
					</th>
					<td>
						<input type="password" id="wpsl-stripe-sk" name="wpsl_stripe_sk" class="regular-text" value="<?php echo esc_attr(wpsl_get_option('wpsl_stripe_sk')); ?>"
							   title="Find in Stripe Dashboard &rarr; Developers &rarr; API Keys &rarr; Secret key (starts with 'sk_'). Use a restricted key if possible."/>
						<p class="description">Stripe Dashboard &rarr; Developers &rarr; API Keys &rarr; Secret key (starts with <code>sk_</code>). Use a restricted key if possible.</p>
						<p class="description">Tip: No webhooks are required; the plugin verifies the Checkout session on return.</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="wpsl-default-currency">Default Currency</label> <span class="wpsl-help"
																									  data-tip="Default currency used for new paywalled streams. You can override per stream under Video access."></span>
					</th>
					<td><input type="text" id="wpsl-default-currency" name="wpsl_default_currency" class="small-text" value="<?php echo esc_attr(wpsl_get_option('wpsl_default_currency', 'usd')); ?>"/> <span
								class="description">ISO currency (e.g. usd, eur)</span>
						<p class="description">Tip: Each stream can override price and currency under Video access.</p>
					</td>
				</tr>
				</tbody>
				<tbody data-wpsl-tab-panel="pages" style="display:none;">
				<tr>
					<th scope="row">Live Channel Page <span class="wpsl-help" data-tip="Public page at /live showing the current live stream using [wpsl_live]."></span></th>
					<td>
						<?php $live_page_id = (int) get_option('wpsl_live_page_id', 0); ?>
						<?php if ($live_page_id && ($p = get_post($live_page_id)) && $p->post_status !== 'trash'): ?>
							<p>Current: <a href="<?php echo esc_url(get_permalink($live_page_id)); ?>" target="_blank" rel="noopener"><?php echo esc_html(get_permalink($live_page_id)); ?></a></p>
						<?php else: ?>
							<p class="description">No Live channel page detected yet.</p>
						<?php endif; ?>
						<p>
							<button class="button" name="wpsl_create_live_page" value="1" type="submit">Create/Update Live Channel Page</button>
							<button class="button" name="wpsl_add_live_to_menu" value="1" type="submit">Add this page to main menu</button>
						</p>
						<p class="description">Creates a public page at <code>/live/</code> with the shortcode <code>[wpsl_live]</code>.</p>
					</td>
				</tr>
				<tr>
					<th scope="row">Videos Page <span class="wpsl-help"
													  data-tip="Public page listing your videos with [wpsl_streams]. Link it in your main menu for easy access. Owned videos show a Purchased badge and Watch now button."></span>
					</th>
					<td>
						<?php $videos_page_id = intval(get_option('wpsl_videos_page_id', 0));
						if (!$videos_page_id) {
							$videos_page_id = intval(get_option('wpsl_streams_page_id', 0));
						}
						if ($videos_page_id && ($p = get_post($videos_page_id)) && $p->post_status !== 'trash'): ?>
							<p>Current: <a href="<?php echo esc_url(get_permalink($videos_page_id)); ?>" target="_blank" rel="noopener"><?php echo esc_html(get_permalink($videos_page_id)); ?></a></p>
						<?php else: ?>
							<p class="description">No Videos page detected yet.</p>
						<?php endif; ?>
						<p>
							<button class="button" name="wpsl_create_videos_page" value="1" type="submit">Create/Update Videos Page</button>
							<button class="button" name="wpsl_add_videos_to_menu" value="1" type="submit">Add this page to main menu</button>
						</p>
						<p class="description">Creates a public page at <code>/videos/</code> with the shortcode <code>[wpsl_streams per_page="12"]</code>.</p>
						<p class="description">Tips: Link this page in your main menu; logged-in users will see a "Purchased" badge on streams they own and a "Watch now" button.</p>
					</td>
				</tr>

				<tr>
					<th scope="row">User Purchase Page <span class="wpsl-help"
															 data-tip="After purchase, buyers receive an email linking here. The page shows all purchased videos with posters and Watch now buttons."></span></th>
					<td>
						<?php $purchases_page_id = (int)get_option('wpsl_purchases_page_id', 0); ?>
						<?php if ($purchases_page_id && ($p = get_post($purchases_page_id)) && $p->post_status !== 'trash'): ?>
							<p>Current: <a href="<?php echo esc_url(get_permalink($purchases_page_id)); ?>" target="_blank" rel="noopener"><?php echo esc_html(get_permalink($purchases_page_id)); ?></a></p>
						<?php else: ?>
							<p class="description">No Purchases page detected yet.</p>
						<?php endif; ?>
						<p>
							<button class="button" name="wpsl_create_purchases_page" value="1" type="submit">Create/Update Purchases Page</button>
							<button class="button" name="wpsl_add_purchases_to_menu" value="1" type="submit">Add this page to main menu</button>
						</p>
						<p class="description">Creates a public page at <code>/my-purchase/</code> with the shortcode <code>[wpsl_my_videos]</code>.</p>
						<p class="description">Tips: Share this link with buyers. After purchase, they receive an email pointing to this page where all purchased videos are listed.</p>
					</td>
				</tr>

				<!-- (Live Channel section moved to the top) -->
				</tbody>
				<tbody data-wpsl-tab-panel="invitation" style="display:none;">
				<tr>
					<th scope="row">Invitation Email</th>
					<td>
						<p style="margin:0 0 8px;"><label><input type="checkbox" id="wpsl-invite-enabled" name="wpsl_invite_enabled" value="1" <?php checked($invite_enabled_val, '1'); ?> /> Enable
								invitation</label></p>
						<div id="wpsl-invite-fields">
							<p style="margin:0 0 6px;"><label for="wpsl-invite-subject">Subject</label><br>
								<input type="text" id="wpsl-invite-subject" name="wpsl_invite_subject" class="regular-text" value="<?php echo esc_attr($invite_subject_val); ?>"/>
							</p>
							<p style="margin:6px 0 6px;"><label for="wpsl-invite-template">Template (HTML)</label></p>
							<?php if (function_exists('wp_editor')): ?>
								<?php wp_editor(
									$invite_template_val,
									'wpsl_invite_template',
									[
										'textarea_name' => 'wpsl_invite_template',
										'textarea_rows' => 10,
										'media_buttons' => false,
										'tinymce' => true,
										'quicktags' => true,
										'editor_height' => 220,
									]
								); ?>
							<?php else: ?>
								<textarea id="wpsl-invite-template" name="wpsl_invite_template" rows="10" class="large-text code"><?php echo esc_textarea($invite_template_val); ?></textarea>
							<?php endif; ?>
							<p class="description">Placeholders: <code>{{viewer_url}}</code>, <code>{{stream_id}}</code>, <code>{{stream_title}}</code>, <code>{{start_time}}</code></p>
						</div>

					</td>
				</tr>
				</tbody>
			</table>
			<p>
				<button class="button button-primary" name="wpsl_save_settings" value="1">Save Settings</button>
			</p>
		</form>
		<script>
			(function () {
				function activate(tab) {
					try {
						localStorage.setItem('wpsl_settings_tab', tab);
					} catch (e) {
					}
					document.querySelectorAll('.nav-tab-wrapper .nav-tab').forEach(function (a) {
						a.classList.toggle('nav-tab-active', a.getAttribute('data-wpsl-tab') === tab);
					});
					document.querySelectorAll('[data-wpsl-tab-panel]').forEach(function (tb) {
						tb.style.display = (tb.getAttribute('data-wpsl-tab-panel') === tab) ? '' : 'none';
					});
				}

				document.querySelectorAll('.nav-tab-wrapper .nav-tab').forEach(function (a) {
					a.addEventListener('click', function (ev) {
						ev.preventDefault();
						activate(a.getAttribute('data-wpsl-tab'));
					});
				});
				var saved = null;
				try {
					saved = localStorage.getItem('wpsl_settings_tab');
				} catch (e) {
				}
				if (saved && document.querySelector('[data-wpsl-tab="' + saved + '"]')) activate(saved); else activate('streaming');
				// Invitation enable toggle
				try {
					var cb = document.getElementById('wpsl-invite-enabled');
					var box = document.getElementById('wpsl-invite-fields');

					function applyInviteState() {
						var on = !!(cb && cb.checked);
						if (!box) return;
						box.style.opacity = on ? '' : '.5';
						box.style.pointerEvents = on ? '' : 'none';
					}

					if (cb) {
						cb.addEventListener('change', applyInviteState);
						applyInviteState();
					}
				} catch (e) {
				}
			})();
		</script>
	</div>
	<?php
}

// Helper function to get option with default
function wpsl_get_option($key, $default = '')
{
	$val = get_option($key);
	return ($val === false || $val === null) ? $default : $val;
}

// Helper function to ensure streams page exists
function wpsl_ensure_streams_page($plugin_instance, bool $force = false): int
{
	$optPrimary = 'wpsl_videos_page_id';
	$optLegacy = 'wpsl_streams_page_id';
	$page_id = intval(get_option($optPrimary, 0));
	if ($page_id <= 0) {
		$legacy = intval(get_option($optLegacy, 0));
		if ($legacy > 0) {
			$page_id = $legacy;
		}
	}
	$content = '[wpsl_streams per_page="12"]';
	$needs_update = false;
	if ($page_id > 0) {
		$p = get_post($page_id);
		if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
			if ($force) {
				$needs_update = true;
			} elseif (strpos((string)$p->post_content, '[wpsl_streams') === false) {
				$needs_update = true;
			}
			if ($needs_update) {
				wp_update_post([
					'ID' => $page_id,
					'post_content' => $content,
					'post_status' => 'publish',
				]);
			}
			// When forcing, also normalize slug/title to 'videos'
			if ($force && $p->post_name !== 'videos') {
				wp_update_post([
					'ID' => $page_id,
					'post_name' => 'videos',
					'post_title' => 'Videos',
				]);
			}
			update_option($optPrimary, $page_id);
			update_option($optLegacy, $page_id);
			return $page_id;
		}
	}

	// Try find by path 'videos' first
	$p = get_page_by_path('videos', OBJECT, 'page');
	if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
		$page_id = (int)$p->ID;
		if ($force || strpos((string)$p->post_content, '[wpsl_streams') === false) {
			wp_update_post([
				'ID' => $page_id,
				'post_content' => $content,
				'post_status' => 'publish',
			]);
		}
		update_option($optPrimary, $page_id);
		update_option($optLegacy, $page_id);
		return $page_id;
	}

	// If only legacy '/streams/' exists, repurpose it by renaming to 'videos'
	$p = get_page_by_path('streams', OBJECT, 'page');
	if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
		$page_id = (int)$p->ID;
		wp_update_post([
			'ID' => $page_id,
			'post_name' => 'videos',
			'post_title' => 'Videos',
			'post_content' => $content,
			'post_status' => 'publish',
		]);
		update_option($optPrimary, $page_id);
		update_option($optLegacy, $page_id);
		return $page_id;
	}

	// Create new page
	$page_id = wp_insert_post([
		'post_type' => 'page',
		'post_status' => 'publish',
		'post_title' => 'Videos',
		'post_name' => 'videos',
		'post_content' => $content,
	]);
	if (!is_wp_error($page_id) && $page_id) {
		update_option($optPrimary, (int)$page_id);
		update_option($optLegacy, (int)$page_id);
	}
	return (int)(is_wp_error($page_id) ? 0 : $page_id);
}

// Helper function to ensure purchases page exists
function wpsl_ensure_purchases_page($plugin_instance, bool $force = false): int
{
	$opt = 'wpsl_purchases_page_id';
	$page_id = (int)get_option($opt, 0);
	$content = '[wpsl_my_videos]';
	if ($page_id > 0) {
		$p = get_post($page_id);
		if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
			if ($force || strpos((string)$p->post_content, '[wpsl_my_videos') === false) {
				wp_update_post(['ID' => $page_id, 'post_content' => $content, 'post_status' => 'publish']);
			}
			update_option($opt, $page_id);
			return $page_id;
		}
	}
	// try find by path
	$p = get_page_by_path('my-purchase', OBJECT, 'page');
	if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
		$page_id = (int)$p->ID;
		if ($force || strpos((string)$p->post_content, '[wpsl_my_videos') === false) {
			wp_update_post(['ID' => $page_id, 'post_content' => $content, 'post_status' => 'publish']);
		}
		update_option($opt, $page_id);
		return $page_id;
	}
	// create new
	$page_id = wp_insert_post([
		'post_type' => 'page', 'post_status' => 'publish', 'post_title' => 'My Purchases', 'post_name' => 'my-purchase', 'post_content' => $content
	]);
	if (!is_wp_error($page_id) && $page_id) update_option($opt, (int)$page_id);
	return (int)(is_wp_error($page_id) ? 0 : $page_id);
}

// Helper function to ensure live channel page exists
function wpsl_ensure_live_page($plugin_instance, bool $force = false): int
{
    $opt = 'wpsl_live_page_id';
    $page_id = (int) get_option($opt, 0);
    $content = '[wpsl_live]';
    if ($page_id > 0) {
        $p = get_post($page_id);
        if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
            if ($force || strpos((string)$p->post_content, '[wpsl_live') === false) {
                wp_update_post([
                    'ID' => $page_id,
                    'post_content' => $content,
                    'post_status' => 'publish',
                ]);
            }
            // Normalize slug/title when forcing
            if ($force && $p->post_name !== 'live') {
                wp_update_post([
                    'ID' => $page_id,
                    'post_name' => 'live',
                    'post_title' => 'Live',
                ]);
            }
            update_option($opt, $page_id);
            return $page_id;
        }
    }
    // Try find by path 'live'
    $p = get_page_by_path('live', OBJECT, 'page');
    if ($p && $p->post_type === 'page' && $p->post_status !== 'trash') {
        $page_id = (int)$p->ID;
        if ($force || strpos((string)$p->post_content, '[wpsl_live') === false) {
            wp_update_post([
                'ID' => $page_id,
                'post_content' => $content,
                'post_status' => 'publish',
            ]);
        }
        update_option($opt, $page_id);
        return $page_id;
    }
    // Create new page
    $page_id = wp_insert_post([
        'post_type' => 'page',
        'post_status' => 'publish',
        'post_title' => 'Live',
        'post_name' => 'live',
        'post_content' => $content,
    ]);
    if (!is_wp_error($page_id) && $page_id) update_option($opt, (int)$page_id);
    return (int)(is_wp_error($page_id) ? 0 : $page_id);
}

// Add a page to the primary/main menu
function wpsl_add_page_to_main_menu(int $page_id): array
{
    $page_id = (int)$page_id;
    if ($page_id <= 0) return [false, 'Page is missing or invalid.'];
    $p = get_post($page_id);
    if (!$p || $p->post_status === 'trash' || $p->post_type !== 'page') return [false, 'Target page is not available.'];

    // Find an existing menu, prefer a menu assigned to a primary-like location
    $menu_id = 0;
    $locations = get_nav_menu_locations(); // location => term_id
    $registered = get_registered_nav_menus(); // location => label
    $prefer = [];
    if (is_array($registered)) {
        if (isset($registered['primary'])) $prefer[] = 'primary';
        if (isset($registered['menu-1'])) $prefer[] = 'menu-1';
        foreach ($registered as $loc => $label) { if (!in_array($loc, $prefer, true)) $prefer[] = $loc; }
    }
    foreach ($prefer as $loc) {
        if (isset($locations[$loc]) && (int)$locations[$loc] > 0) { $menu_id = (int)$locations[$loc]; break; }
    }
    if ($menu_id <= 0) {
        // Fallback: first existing menu
        $menus = wp_get_nav_menus();
        if (!empty($menus)) $menu_id = (int)$menus[0]->term_id;
    }
    if ($menu_id <= 0) {
        // Create a basic menu
        $menu_id = (int) wp_create_nav_menu('Main Menu');
        // Assign it to a primary-like location if possible
        if (is_array($registered) && !empty($registered)) {
            $locs = get_theme_mod('nav_menu_locations');
            if (!is_array($locs)) $locs = [];
            $assign_to = 'primary';
            if (!isset($registered[$assign_to])) {
                $assign_to = isset($registered['menu-1']) ? 'menu-1' : array_key_first($registered);
            }
            if ($assign_to) { $locs[$assign_to] = $menu_id; set_theme_mod('nav_menu_locations', $locs); }
        }
    }
    if ($menu_id <= 0) return [false, 'Could not determine a menu to update.'];

    // Avoid duplicate
    $items = wp_get_nav_menu_items($menu_id);
    if (is_array($items)) {
        foreach ($items as $it) {
            if ((int)$it->object_id === $page_id && $it->object === 'page') {
                return [true, 'Page is already in the main menu.'];
            }
        }
    }

    $item_id = wp_update_nav_menu_item($menu_id, 0, [
        'menu-item-title'     => get_the_title($page_id) ?: 'Page',
        'menu-item-object'    => 'page',
        'menu-item-object-id' => $page_id,
        'menu-item-type'      => 'post_type',
        'menu-item-status'    => 'publish',
    ]);
    if (is_wp_error($item_id) || !$item_id) return [false, 'Failed to add page to the menu.'];
    return [true, 'Added to main menu.'];
}
